/** \file
 * \brief IUP binding for Lua 5.
 *
 * See Copyright Notice in iup.h
 * $Id: iuplua.c,v 1.28 2004/09/29 14:39:37 scuri Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <iup.h>
#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>
#include <assert.h>

#include "iuplua.h"
#include "il.h"

/* Iup Modules */
#include "luaalarm.h"
#include "luabutton.h"
#include "luacanvas.h"
#include "luadialog.h"
#include "luafill.h"
#include "luaframe.h"
#include "luafiledlg.h"
#include "luahbox.h"
#include "luaitem.h"
#include "luaimage.h"
#include "lualabel.h"
#include "lualist.h"
#include "lualistdialog.h"
#include "luamenu.h"
#include "luamultiline.h"
#include "luaradio.h"
#include "luaseparator.h"
#include "luasubmenu.h"
#include "luatext.h"
#include "luatoggle.h"
#include "luavbox.h"
#include "luazbox.h"
#include "luatimer.h"
#include "scanf.h"

static Ihandle *il_checkihandleornil(lua_State *L, int pos);

/*****************************************************************************
 * Auxiliary functions                                                       *
 ****************************************************************************/
#define lua_isnull(L,n)           (lua_type(L,n) == LUA_TNONE)
const char * il_optstring(lua_State *L, int n, const char *def)
{
    if (lua_isnil(L,n) || lua_isnull(L,n)) /* removed */
        return def;
    else
        return luaL_check_string(L, n);
}


Ihandle * il_plugstate(lua_State *L, Ihandle *h)
{
    IupSetAttribute(h, "lua5_state_context",(char *) L);
    return h;    
}

lua_State * il_getL(Ihandle *h)
{
     return (lua_State *) IupGetAttribute(h, "lua5_state_context");
}

int il_cleanup(lua_State *L)
{
    int tmp = (int) lua_isnil(L, -1) ? IUP_DEFAULT : (int)lua_tonumber(L,-1);
    lua_pop(L, 1);
    return tmp;    
}


char * il_strdup_(const char *s)
{
  char *t = malloc(strlen(s) + 1);
  if (t) strcpy(t, s);
  return t;
}

static Ihandle *il_checkihandleornil(lua_State *L, int pos)
{
    lua_getmetatable(L, pos);
    lua_pushstring(L, "iup handle");
    lua_gettable(L, LUA_REGISTRYINDEX);
    if (lua_equal(L, -2, -1)) {
        lua_pop (L, 2);
        return (Ihandle *) lua_unboxpointer(L, pos);
    } else {
        return NULL;
    }
}

Ihandle *il_checkihandle(lua_State *L, int pos)
{
    lua_getmetatable(L, pos);
    lua_pushstring(L, "iup handle");
    lua_gettable(L, LUA_REGISTRYINDEX);
    if (lua_equal(L, -2, -1)) {
        lua_pop (L, 2);
        return (Ihandle *) lua_unboxpointer(L, pos);
    } else {
        luaL_argerror(L, pos, "iup handle expected");
        return NULL;
    }
}

void il_pushihandle(lua_State *L, Ihandle *h)
{
    if (h) {
        lua_boxpointer(L, h);
        lua_pushstring(L, "iup handle");
        lua_gettable(L, LUA_REGISTRYINDEX);
        lua_setmetatable(L, -2);
    } else lua_pushnil(L);
}

static int luaCompareIhandle(lua_State *L)
{
  Ihandle *a = il_checkihandle(L, 1);
  Ihandle *b = il_checkihandle(L, 2);
  if(a == b)
    lua_pushboolean(L, 1);
  else
    lua_pushboolean(L, 0);
  return 1;
}

static int c_GetFromC(lua_State *L)
{
  Ihandle *n;
  const char *a;
  if (!lua_istable(L, -1)) 
  {
    lua_pushstring(L, "IupGetFromC: wrong arguments to function"); 
    lua_error(L);
    return 0;
  }
  lua_pushnumber(L, 1);
  lua_gettable(L, -2);
  if (!lua_isstring(L, -1)) 
  {
    lua_pushstring(L, "IupGetFromC: wrong arguments to function"); 
    lua_error(L);
    return 0;
  }
  a = lua_tostring(L, -1);
  n = IupGetHandle((char*)a);
  if(n)
    il_pushihandle(L, n);
  else
    lua_pushnil(L);
  return 1;
}

void il_pushstring(lua_State *L, const char *s)
{
    if (s) lua_pushstring(L,s);
    else lua_pushnil(L);
}

char ** il_stringvectorfromtable(lua_State *L, int pos)
{
    int i,n;
    char **v;
    n = luaL_getn(L,pos);
    v = (char **) malloc (n*sizeof(char *));
    for(i=1; i<=n; i++)
    {
        lua_pushnumber(L,i);
        lua_gettable(L,pos);
        v[i-1] = (char *) malloc (lua_strlen(L,-1)+1);
        strcpy(v[i-1], lua_tostring(L, -1));
        lua_pop(L,1);
    }
    return v;
}

int * il_intvectorfromtable(lua_State *L, int pos)
{
    int i,n;
    int *v;
    n = luaL_getn(L,pos);
    v = (int *) malloc (n*sizeof(int));
    for(i=1; i<=n; i++)
    {
        lua_pushnumber(L,i);
        lua_gettable(L,pos);
        v[i-1] = (int)lua_tonumber(L, -1);
        lua_pop(L,1);
    }
    return v;
}

Ihandle ** il_ihandlevectorfromtable(lua_State *L, int pos)
{
    Ihandle **v;
    int i, n = luaL_getn(L, pos);
    v = (Ihandle **) malloc (n*sizeof(Ihandle *));
    for (i=1; i<=n; i++)
    {
        lua_pushnumber(L,i);
        lua_gettable(L,pos);
        v[i-1] = il_checkihandle(L, -1);
        lua_pop(L,1);
    }
    v[i-1] = NULL;
    return v;
}

/* Called when a callback in Lua should be called. If error, 
 * _ERRORMESSAGE is called. */
int iuplua_call(lua_State* L, int nargs, int nresults)
{
  int result = 0;

  lua_getglobal(L, "iup");
  lua_pushstring(L, "_ERRORMESSAGE");
  lua_gettable(L, -2);
  lua_remove(L, -2);
  lua_insert(L, 1);

  result = lua_pcall(L, nargs, nresults, 1);

  lua_remove(L, 1);
  
  return result;
}

static void print_message(lua_State *L, const char *msg) 
{
  lua_getglobal(L, "iup");
  lua_pushstring(L, "_ERRORMESSAGE");
  lua_gettable(L, -2);
  lua_remove(L, -2);
  if(lua_isnil(L, -1))
  {
    printf("%s\n", msg); /* Panic mode */
    return;
  }
  lua_pushstring(L, msg);
  lua_call(L, 1, 0);
}


static int report (lua_State *L, int status) 
{
  const char *msg;
  if (status) 
  {
    msg = lua_tostring(L, -1);
    if (msg == NULL) msg = "(error with no message)";
    print_message(L, msg);
    lua_pop(L, 1);
  }
  return status;
}

static int lcall (lua_State *L, int narg, int clear) {
  int status;
  int base = lua_gettop(L) - narg;  /* function index */
  lua_pushliteral(L, "_TRACEBACK");
  lua_rawget(L, LUA_GLOBALSINDEX);  /* get traceback function */
  lua_insert(L, base);  /* put it under chunk and args */
  status = lua_pcall(L, narg, (clear ? 0 : LUA_MULTRET), base);
  lua_remove(L, base);  /* remove traceback function */
  return status;
}


static int docall (lua_State* L, int status) 
{
  if (status == 0) 
    status = lcall(L, 0, 1);
  return report(L, status);
}


void iuplua_pushihandle(lua_State *L, Ihandle *n)
{
  il_pushihandle(L, n);
}

int iuplua_dofile(lua_State *L, char *filename)
{
  return docall(L, luaL_loadfile(L, filename));
}

int iuplua_dostring(lua_State *L, const char *s, const char *name)
{
  return docall(L, luaL_loadbuffer(L, s, strlen(s), name));
}

/*****************************************************************************
 * Iuplua bind functions                                                     *
 ****************************************************************************/




/*-- FUNCTIONS -------------------------------------------------------------*/

/*****************************************************************************
 * Append                                                                    *
 ****************************************************************************/

static int Append(lua_State *L)
{
  Ihandle *box, *element,*res;
  box = il_checkihandle(L, 1);
  element = il_checkihandle(L,2);
  res = IupAppend(box,element);
  il_pushihandle(L,res);
  return 1;
}

/*****************************************************************************
 * Destroy                                                                   *
 ****************************************************************************/
static int Destroy(lua_State *L)
{
  Ihandle *element = il_checkihandle(L,1);
  IupDestroy(element);
  return 0;
}

/*****************************************************************************
 * Detach                                                                   *
 ****************************************************************************/
static int Detach(lua_State *L)
{
  Ihandle *element = il_checkihandle(L,1);
  IupDetach(element);
  return 0;
}

/*****************************************************************************
 * Flush                                                                    *
 ****************************************************************************/
static int Flush(lua_State *L)
{
  IupFlush();
  return 0;
}

/*****************************************************************************
 * GetAttributeData                                                         *
 ****************************************************************************/
static int GetAttributeData (lua_State *L)
{
  Ihandle *element = il_checkihandle(L,1);
  const char *attribute = luaL_check_string(L,2);
  const char *value = IupGetAttribute(element, (char *)attribute);
  if (!value)
    lua_pushnil(L);
  else
    lua_pushlightuserdata(L, (void*)value);
  return 1;
}

/*****************************************************************************
 * GetAttribute                                                              *
 ****************************************************************************/
static int GetAttribute (lua_State *L)
{
  Ihandle *element = il_checkihandle(L,1);
  const char *attribute = luaL_check_string(L,2);
  const char *value = IupGetAttribute(element, (char *)attribute);
  if (!value)
    lua_pushnil(L);
  else
  {
    if (iupIsPointer(attribute))
      lua_pushlightuserdata(L, (void*)value);
    else
      lua_pushstring(L,value);
  }
  return 1;
}

/*****************************************************************************
* GetAttributes                                                             *
****************************************************************************/
static int GetAttributes(lua_State *L)
{
  Ihandle *element = il_checkihandle(L,1);
  const char *value = IupGetAttributes(element);
  il_pushstring(L,value);
  return 1;
}

/*****************************************************************************
* GetAllDialogs                                                             *
****************************************************************************/
static int GetAllDialogs(lua_State *L)
{
  int n = (int)luaL_check_number(L,2);
  char **names = (char **) malloc (n * sizeof(char *));
  int i,ret;
  if (!names)
      lua_pushnil(L);
  ret = IupGetAllDialogs(names,n);
  for (i=0; i<ret; i++)
  {
      lua_pushnumber(L,i+1);
      lua_pushstring(L,names[i]);
      lua_settable(L,1);
  }
  lua_pushnumber(L,ret);
  return 1;
}

/*****************************************************************************
* GetDialog                                                                 *
****************************************************************************/
static int GetDialog(lua_State *L)
{
  Ihandle *h1 = il_checkihandle(L,1);
  Ihandle *h2 = IupGetDialog(h1);
  il_pushihandle(L,h2);
  return 1;
}

/*****************************************************************************
* GetFile                                                                   *
****************************************************************************/
static int GetFile (lua_State *L)
{
  const char *fname = luaL_check_string(L,1);
  char returned_fname[FILENAME_MAX];
  int ret;
  strcpy(returned_fname, fname);
    ret = IupGetFile((char *) returned_fname);
  lua_pushstring(L, returned_fname);
  lua_pushnumber(L, ret);
  return 2;
}

/*****************************************************************************
* GetFocus                                                                  *
****************************************************************************/
static int GetFocus(lua_State *L)
{
  Ihandle *h = IupGetFocus();
  il_pushihandle(L,h);
  return 1;
}

/*****************************************************************************
* GetType                                                                  *
****************************************************************************/
static int GetType(lua_State *L)
{
  Ihandle *h = il_checkihandle(L, 1);
  il_pushstring(L, IupGetType(h));
  return 1;
}

/*****************************************************************************
* GetGlobal                                                                 *
****************************************************************************/
static int GetGlobal(lua_State *L)
{
  char *a = (char *) luaL_check_string(L,1);
  const char *v = IupGetGlobal(a);
  il_pushstring(L,v);
  return 1;
}

/*****************************************************************************
* GetHandle                                                                 *
****************************************************************************/
static int GetHandle(lua_State *L)
{
  const char *name;
  Ihandle *h;
  name = luaL_check_string(L,1);
  h = IupGetHandle((char *)name);
  il_pushihandle(L,h);
  return 1;
}

/*****************************************************************************
* GetLanguage                                                               *
****************************************************************************/
static int GetLanguage (lua_State *L)
{
  char * value;
  value = IupGetLanguage();
  il_pushstring(L,value);
  return 1;
}

/*****************************************************************************
* GetName                                                                   *
****************************************************************************/
static int GetName(lua_State *L)
{
  Ihandle *h = il_checkihandle(L,1);
  char * name = IupGetName(h);
  il_pushstring(L,name);
  return 1;
}

/*****************************************************************************
* Help                                                                      *
****************************************************************************/
static int Help(lua_State *L)
{
  char *s = (char *) luaL_check_string(L,1);
  IupHelp(s);
  return 0;
}

/*****************************************************************************
* Hide                                                                      *
****************************************************************************/
static int Hide(lua_State *L)
{
  Ihandle *h = il_checkihandle(L,1);
  lua_pushnumber(L, IupHide(h));
  return 1;
}

/*****************************************************************************
* Load                                                                      *
****************************************************************************/
static int Load(lua_State *L)
{
  char *s = (char *) luaL_check_string(L,1);
  const char *r = IupLoad(s);
  il_pushstring(L,r);
  return 1;
}

/*****************************************************************************
* LoopStep                                                                  *
****************************************************************************/
static int LoopStep(lua_State *L)
{
  lua_pushnumber(L,IupLoopStep());
  return 1;
}

/*****************************************************************************
* MainLoop                                                                  *
****************************************************************************/
static int MainLoop(lua_State *L)
{
  lua_pushnumber(L,IupMainLoop());
  return 1;
}

/*****************************************************************************
* Map                                                                       *
****************************************************************************/
static int Map(lua_State *L)
{
  Ihandle *h = il_checkihandle(L,1);
  lua_pushnumber(L, IupMap(h));
  return 1;
}

/*****************************************************************************
* MapFont                                                                   *
****************************************************************************/
static int MapFont(lua_State *L)
{
  char *font = (char *) luaL_check_string(L,1);
  const char *nfont = IupMapFont(font);
  il_pushstring(L, nfont);
  return 1;
}

/*****************************************************************************
* Message                                                                   *
****************************************************************************/
static int Message(lua_State *L)
{
  char *title = (char *) luaL_check_string(L,1);
  char *message = (char *) luaL_check_string(L,2);
  IupMessage(title, message);
  return 0;
}

/*****************************************************************************
* NextField                                                                 *
****************************************************************************/
static int NextField(lua_State *L)
{
  Ihandle *h1 = il_checkihandle(L,1);
  Ihandle *h2 = IupNextField(h1);
  il_pushihandle(L,h2);
  return 1;
}

/*****************************************************************************
* PreviousField                                                             *
****************************************************************************/
static int PreviousField(lua_State *L)
{
  Ihandle *h1 = il_checkihandle(L,1);
  Ihandle *h2 = IupNextField(h1);
  il_pushihandle(L,h2);
  return 1;
}

/*****************************************************************************
* Popup                                                                     *
****************************************************************************/
static int Popup(lua_State *L)
{
  Ihandle *h = il_checkihandle(L,1);
  int x = (int)luaL_check_number(L,2);
  int y = (int)luaL_check_number(L,3);
  lua_pushnumber(L,IupPopup(h,x,y));
  return 1;
}

/*****************************************************************************
 * cfXCODE                                                                   *
 *****************************************************************************/
static int cfXCODE(lua_State *L)
{
  int value = (int) luaL_check_number(L, 1);
  lua_pushnumber(L, xCODE(value));
  return 1;
}

/*****************************************************************************
 * cfisbutton1                                                                 *
 ****************************************************************************/
static int cfisbutton1(lua_State *L)
{
  const char *value = luaL_check_string(L, 1);
  if (isbutton1(value)) lua_pushnumber(L, 1);
  else lua_pushnil(L);
  return 1;
}

/*****************************************************************************
 * cfisbutton2                                                                 *
 ****************************************************************************/
static int cfisbutton2(lua_State *L)
{
  const char *value = luaL_check_string(L, 1);
  if (isbutton2(value)) lua_pushnumber(L, 1);
  else lua_pushnil(L);
  return 1;
}

/*****************************************************************************
 * cfisbutton3                                                                 *
 ****************************************************************************/
static int cfisbutton3(lua_State *L)
{
  const char *value = luaL_check_string(L, 1);
  if (isbutton3(value)) lua_pushnumber(L, 1);
  else lua_pushnil(L);
  return 1;
}

/*****************************************************************************
 * cfisshift                                                                 *
 ****************************************************************************/
static int cfisshift(lua_State *L)
{
  const char *value = luaL_check_string(L, 1);
  if (isshift(value)) lua_pushnumber(L, 1);
  else lua_pushnil(L);
  return 1;
}

/*****************************************************************************
 * cfiscontrol                                                                 *
 ****************************************************************************/
static int cfiscontrol(lua_State *L)
{
  const char *value = luaL_check_string(L, 1);
  if (iscontrol(value)) lua_pushnumber(L, 1);
  else lua_pushnil(L);
  return 1;
}

/*****************************************************************************
 * cfisdouble                                                                 *
 ****************************************************************************/
static int cfisdouble(lua_State *L)
{
  const char *value = luaL_check_string(L, 1);
  if (isdouble(value)) lua_pushnumber(L, 1);
  else lua_pushnil(L);
  return 1;
}

/*****************************************************************************
 * IupGetParent                                                              *
 ****************************************************************************/
static int GetParent(lua_State *L)
{
  Ihandle * element = il_checkihandle(L,1);
  Ihandle * parent = IupGetParent(element);
  il_pushihandle(L, parent);
  return 1;
}

/*****************************************************************************
 * IupVersionNumber
 ****************************************************************************/
static int VersionNumber(lua_State *L)
{
  lua_pushnumber(L, IupVersionNumber());
  return 1;
}

/*****************************************************************************
 * IupGetNextChild
 ****************************************************************************/
static int GetNextChild(lua_State *L)
{
  Ihandle * parent = il_checkihandle(L,1);
  Ihandle * next = il_checkihandleornil(L,2);
  Ihandle * nextchild = IupGetNextChild(parent, next);
  il_pushihandle(L, nextchild);
  return 1;
}

/*****************************************************************************
 * IupGetBrother                                                              *
 ****************************************************************************/
static int GetBrother(lua_State *L)
{
  Ihandle * element = il_checkihandle(L,1);
  Ihandle * brother = IupGetBrother(element);
  il_pushihandle(L, brother);
  return 1;
}

/*****************************************************************************
 * il_pushtable                                                              *
 ****************************************************************************/
void il_pushtable(lua_State *L, Ihandle *element)
{
  int ref = (int) IupGetAttribute(element, "iuplua_object_table");
  lua_getref(L, ref);
}

/*****************************************************************************
 * iupGetTable                                                              *
 ****************************************************************************/
static int GetTable(lua_State *L)
{
  Ihandle * element = il_checkihandle(L,1);
  il_pushtable(L, element);
  return 1;
}

/*****************************************************************************
 * iupSetTable                                                              *
 ****************************************************************************/
static int SetTable(lua_State *L)
{
  Ihandle * element = il_checkihandle(L,1);
  int ref = lua_ref(L, 1);
  IupSetAttribute(element, "iuplua_object_table", (char *) ref);
  return 0;
}

/*****************************************************************************
 * SetAttributes                                                             *
 ****************************************************************************/
static int SetAttributes(lua_State *L)
{
    Ihandle * element = il_checkihandle(L,1);
    char *attributes = (char *) luaL_check_string(L,2);
    Ihandle * h = IupSetAttributes(element, attributes);
    il_pushihandle(L,h);
    return 1;
}

/*****************************************************************************
 * SetFocus                                                                  *
 ****************************************************************************/
static int SetFocus(lua_State *L)
{
    Ihandle *h1 = il_checkihandle(L,1);
    Ihandle *h2 = IupSetFocus(h1);
    il_pushihandle(L,h2);
    return 1;
}

/*****************************************************************************
 * SetGlobal                                                                 *
 ****************************************************************************/
static int SetGlobal(lua_State *L)
{
    char *a = (char *) luaL_check_string(L,2);
    char *v = (char *) luaL_check_string(L,3);
    IupSetGlobal(a,v);
    return 0;
}

/*****************************************************************************
 * SetHandle                                                                 *
 ****************************************************************************/
static int SetHandle(lua_State *L)
{
	  Ihandle *last = NULL;
    const char *name;
    Ihandle *element;
    name = luaL_check_string(L,1);
    element = il_checkihandle(L,2);
    last = IupSetHandle(il_strdup_(name), element);
		il_pushihandle(L, last);
    return 1;
}

/*****************************************************************************
 * Idlecall                                                                  *
 ****************************************************************************/
static int Idlecall()
{
    int ret = 0;
    lua_State *L = (lua_State *) IupGetGlobal("_IUP_LUA_DEFAULT_STATE");
    lua_getglobal(L, "_IUP_LUA_IDLE_FUNC_"); 
    lua_call(L, 0, 1);
    ret = (int) lua_tonumber(L, -1);
    lua_pop(L, 1);
    return ret;
}

/*****************************************************************************
 * SetIdle                                                                   *
 ****************************************************************************/
static int SetIdle(lua_State *L)
{
    if lua_isnil(L,1)
        IupSetFunction(IUP_IDLE_ACTION, NULL);
    else
    {
        lua_pushvalue(L,1);
        lua_setglobal(L, "_IUP_LUA_IDLE_FUNC_");
        IupSetFunction(IUP_IDLE_ACTION, (Icallback) Idlecall);
    }
    return 0;
}

/*****************************************************************************
 * SetLanguage                                                               *
 ****************************************************************************/
static int SetLanguage(lua_State *L)
{
    char *name = (char *) luaL_check_string(L,1);
    IupSetLanguage(name);
    return 0;
}

/*****************************************************************************
 * Show                                                                      *
 ****************************************************************************/
static int Show (lua_State *L)
{
    Ihandle * element;
    element = il_checkihandle(L,1);
    lua_pushnumber(L, IupShow(element));
    return 1;
}

/*****************************************************************************
 * ShowXY                                                                    *
 ****************************************************************************/
static int ShowXY(lua_State *L)
{
    Ihandle *h = il_checkihandle(L,1);
    int x = (int)luaL_check_number(L,2);
    int y = (int)luaL_check_number(L,3);
    lua_pushnumber(L,IupShowXY(h,x,y));
    return 1;
}

/*****************************************************************************
 * StoreAttribute                                                            *
 ****************************************************************************/
static int StoreAttribute(lua_State *L)
{
    Ihandle *h = il_checkihandle(L,1);
    char *a = (char *) luaL_check_string(L,2);
    if (lua_isnil(L,3)) IupStoreAttribute(h,a,NULL);
    else 
    {
		char *v;
		if(lua_isuserdata(L,3)) {
            v = (char *) lua_touserdata(L,3);
			IupSetAttribute(h,a,v);
		}
		else {
            v = (char *) luaL_check_string(L,3);
			IupStoreAttribute(h,a,v);
		}
    }
    return 0;
}

/*****************************************************************************
 * StoreGlobal                                                               *
 ****************************************************************************/
static int StoreGlobal(lua_State *L)
{
    char *a = (char *) luaL_check_string(L,2);
    char *v = (char *) luaL_check_string(L,3);
    IupStoreGlobal(a,v);
    return 0;
}

/*****************************************************************************
 * UnMapFont                                                                 *
 ****************************************************************************/
static int UnMapFont (lua_State *L)
{
    char *s = (char *) luaL_check_string(L,1);
    char *n = (char *) IupUnMapFont(s);
    il_pushstring(L,n);
    return 1;
}

/*
* Class manipulation functions
*/
static int luaSetClass(lua_State *L)
{
    int ret = 0;
    if (lua_isnil(L, -1)) /* no name in function, just ignore */
      return 0;
    lua_gettable(L, LUA_REGISTRYINDEX);
    if (lua_isnil(L, -1))
    {
    	lua_pushstring(L, "invalid class name");
      lua_error(L);
    }
    ret = lua_setmetatable(L, -2);
    assert(ret != 0);
    return 0;
}

static int luaNewClass(lua_State *L)
{
    lua_newtable(L);
    lua_pushstring(L, "class");
    lua_pushvalue(L, 1);
    lua_settable(L, -3);
    lua_settable(L, LUA_REGISTRYINDEX);
    return 0;
}

static int luaClass(lua_State *L)
{
    if (lua_istable(L, 1) || lua_isuserdata(L, 1)) {
        lua_getmetatable(L, 1);
        if (lua_istable(L, -1)) {
            lua_pushstring(L, "class");
            lua_gettable(L, -2);
            return 1;
        }
    }
    lua_pushnil(L);
    return 1;
}

static int luaSetMethod(lua_State *L)
{
    lua_pushvalue(L, 1);
    lua_gettable(L, LUA_REGISTRYINDEX);
    if (lua_isnil(L, -1)) 
    {
    	lua_pushstring(L, "invalid class name");
    	lua_error(L);
    }
    lua_pushvalue(L, -3);
    lua_pushvalue(L, -3);
    lua_settable(L, -3);
    return 0;
}

void iuplua_changeEnv(lua_State *L)
{
  lua_pushvalue(L, LUA_GLOBALSINDEX);
  lua_getglobal(L, "iup");
  lua_replace(L, LUA_GLOBALSINDEX); /* Replaces global environment */
}

void iuplua_returnEnv(lua_State *L)
{
  /* reestablishes global environment */
  lua_replace(L, LUA_GLOBALSINDEX);
}

/*--LUA_OPEN----------------------------------------------------------------*/

void iuplua_open(lua_State * L)
{
    struct luaL_reg funcs[] = {
        {"Append", Append},
        {"Destroy", Destroy},
        {"Detach", Detach},
        {"Flush", Flush},
        {"GetAttribute", GetAttribute},
        {"GetAttributeData", GetAttributeData},
        {"GetAttributes", GetAttributes},
        {"GetAllDialogs", GetAllDialogs},
        {"GetDialog", GetDialog},
        {"GetFile", GetFile},
        {"GetFocus", GetFocus},
        {"GetType", GetType},
        {"GetGlobal", GetGlobal},
        {"GetHandle", GetHandle},
        {"GetLanguage", GetLanguage},
        {"GetName", GetName},
        {"Help", Help},
        {"Hide", Hide},
        {"Load", Load},
        {"LoopStep", LoopStep},
        {"MainLoop", MainLoop},
        {"Map", Map},
        {"MapFont", MapFont},
        {"Message", Message},
        {"NextField", NextField},
        {"Popup", Popup},
        {"PreviousField", PreviousField},
        {"SetAttribute", StoreAttribute},
        {"SetAttributes", SetAttributes},
        {"isbutton1", cfisbutton1},
        {"isbutton2", cfisbutton2},
        {"isbutton3", cfisbutton3},
        {"isshift", cfisshift},
        {"iscontrol", cfiscontrol},
        {"isdouble", cfisdouble},
        {"GetParent", GetParent},
        {"GetNextChild", GetNextChild},
        {"VersionNumber", VersionNumber},
        {"GetBrother", GetBrother},
        {"SetFocus", SetFocus},
        {"SetGlobal", SetGlobal},
        {"SetHandle", SetHandle},
        {"SetIdle", SetIdle},
        {"SetLanguage", SetLanguage},
        {"Show", Show},
        {"ShowXY", ShowXY},
        {"StoreAttribute", StoreAttribute},
        {"StoreGlobal", StoreGlobal},
        {"UnMapFont", UnMapFont},
    		{"Scanf", Scanf},
    		{"GetFromC", c_GetFromC},
        {"xCODE", cfXCODE},
        {"iupGetTable", GetTable},
        {"iupSetTable", SetTable},
        {"iupluaSetClass", luaSetClass},
        {"iupluaNewClass", luaNewClass},
        {"iupluaSetMethod", luaSetMethod},
        {"iupluaClass", luaClass},
    		{"iupluaCompareIhandle", luaCompareIhandle},
        {NULL, NULL},
    };

    /* Creating global namespace iup */
    lua_newtable(L);
    lua_setglobal(L, "iup");

    /* Registers functions in iup namespace */
    luaL_openlib(L, "iup", funcs, 0);

    /* Creating new environment with metamethod __index (so that normal
     * Lua functions will be found) */
    iuplua_dostring(L, "iup._G = _G\
      setmetatable(iup, {__index = iup._G})", "iupluasetmatatable");

    /* Changing environment to iup (all created variables from now on
     * will be put in iup environment) */
    iuplua_changeEnv(L);

    /* create basic iup class */
    lua_pushcfunction(L, luaNewClass);
    lua_pushstring(L, "iup handle");
    lua_call(L, 1, 0);

    IupSetGlobal("_IUP_LUA_DEFAULT_STATE", (char *) L);

#ifdef IUPLUA_USELOH
#include "iuplua.loh"
#include "constants.loh"
#else
    lua_dofile(L, "iuplua.lua");
    lua_dofile(L, "constants.lua");
#endif

    iuplua_returnEnv(L);
  
    /* Iup Modules initialization */
    alarmlua_open(L);
    buttonlua_open(L);
    canvaslua_open(L);
    dialoglua_open(L);
    filllua_open(L);
    framelua_open(L);
    filedlglua_open(L);
    hboxlua_open(L);
    itemlua_open(L);
    imagelua_open(L);
    labellua_open(L);
    listlua_open(L);
    listdialoglua_open(L);
    menulua_open(L);
    multilinelua_open(L);
    radiolua_open(L);
    separatorlua_open(L);
    submenulua_open(L);
    textlua_open(L);
    togglelua_open(L);
    vboxlua_open(L);
    zboxlua_open(L);
    timerlua_open(L);
}

int iupkey_open(lua_State *L)
{
  iuplua_changeEnv(L);
#ifdef IUPLUA_USELOH
#include "iupkey.loh"
#else
  lua_dofile(L, "iupkey.lua");
#endif
  iuplua_returnEnv(L);
  return 1;
}

